document.addEventListener('DOMContentLoaded', () => {
    const spread = document.querySelector('.tarot-spread');
    if (!spread) return; // Exit if not on card selection page

    // --- Constants ---
    const TOTAL_CARDS = 78;
    const CARDS_TO_PICK = 3;
    const REVERSAL_PROBABILITY = 0.3; // 30% chance of reversal
    const REPETITIONS = 888;
    const HASH_LEVELS = 888;
    const HASH_DISPLAY_LENGTH = 8; // Show first 8 chars of hash

    // --- DOM Elements ---
    const cardGrid = document.getElementById('cardGrid');
    const selectedCards = document.getElementById('selectedCards');
    const resetButton = document.getElementById('resetButton');

    // --- State ---
    let question = spread.dataset.question;
    let deckHash = spread.dataset.deckHash;
    let initialHash = spread.dataset.initialHash;
    let timeSeed = parseInt(spread.dataset.timeSeed, 10);
    let selectedCount = 0;
    let cardStates = new Array(TOTAL_CARDS).fill(null); // Tracks card indices, reversal, and hash
    let deck = [];

    // --- Initialization ---
    fetch('cards.json')
        .then(response => response.json())
        .then(data => {
            deck = data;
            initializeSpread();
        })
        .catch(error => console.error('Error loading card data:', error));

    // --- Helper Functions ---
    async function sha256(str) {
        const buffer = new TextEncoder().encode(str);
        const hashBuffer = await crypto.subtle.digest('SHA-256', buffer);
        const hashArray = Array.from(new Uint8Array(hashBuffer));
        return hashArray.map(b => b.toString(16).padStart(2, '0')).join('');
    }

    async function multiHash(segment) {
        let hashValue = segment.repeat(REPETITIONS);
        for (let i = 0; i < HASH_LEVELS; i++) {
            hashValue = await sha256(hashValue);
            hashValue = hashValue.repeat(REPETITIONS);
            if (i % 50 === 0 && HASH_LEVELS > 100) {
                await new Promise(resolve => setTimeout(resolve, 0));
            }
        }
        return hashValue;
    }

    function hashToNumber(hash, max) {
        const hashNum = parseInt(hash.substring(0, 8), 16);
        return hashNum % max;
    }

    // --- Card Selection Logic ---
    async function calculateCardIndex(iteration) {
        const segment = `${question}:${timeSeed}:${iteration}`;
        const hash = await multiHash(segment);
        return { index: hashToNumber(hash, TOTAL_CARDS), hash };
    }

    function initializeSpread() {
        cardGrid.innerHTML = '';
        cardStates = new Array(TOTAL_CARDS).fill(null);
        selectedCount = 0;
        resetButton.classList.add('hidden');

        // Create all 78 cards in a single overlapping row
        const baseOffset = 20; // Base horizontal offset for overlap
        for (let i = 0; i < TOTAL_CARDS; i++) {
            const cardDiv = document.createElement('div');
            cardDiv.className = 'card card-back';
            cardDiv.dataset.index = i;
            cardDiv.style.left = `${i * (baseOffset * 0.7)}px`; // Gradual overlap
            cardDiv.style.zIndex = TOTAL_CARDS - i; // Higher z-index for cards on the left
            cardDiv.addEventListener('click', () => selectCard(i));
            cardGrid.appendChild(cardDiv);
        }
    }

    async function selectCard(spreadIndex) {
        if (selectedCount >= CARDS_TO_PICK || cardStates[spreadIndex]) return;

        selectedCount++;
        const iteration = selectedCount; // Unique iteration for each card
        const { index: cardIndex, hash } = await calculateCardIndex(iteration);
        const isReversed = Math.random() < REVERSAL_PROBABILITY;
        const shortHash = hash.substring(0, HASH_DISPLAY_LENGTH);
        cardStates[spreadIndex] = { cardIndex, isReversed, hash: shortHash };

        // Update card in grid
        const cardDiv = cardGrid.children[spreadIndex];
        cardDiv.classList.remove('card-back');
        cardDiv.classList.add('card-revealed');
        cardDiv.style.backgroundImage = `url(${deck[cardIndex].image})`;
        if (isReversed) {
            cardDiv.classList.add('reversed');
        }
        // Add vertical hash to card
        const hashDiv = document.createElement('div');
        hashDiv.className = 'card-hash';
        hashDiv.textContent = shortHash.split('').join('\n'); // Split hash chars for vertical display
        cardDiv.appendChild(hashDiv);

        // Add to selected cards display
        const cardInfo = document.createElement('div');
        cardInfo.className = 'selected-card';
        cardInfo.innerHTML = `
            <h3>${deck[cardIndex].name} (${isReversed ? 'Reversed' : 'Upright'})</h3>
            <img src="${deck[cardIndex].image}" alt="${deck[cardIndex].name}" class="${isReversed ? 'reversed' : ''}">
            <p><strong>Energy Hash:</strong> ${shortHash}</p>
            <p><strong>Overall Meaning:</strong> ${deck[cardIndex].meaning}</p>
            <p><strong>${isReversed ? 'Reversed' : 'Upright'} Interpretation:</strong> ${isReversed ? deck[cardIndex].rdesc : deck[cardIndex].desc}</p>
        `;
        selectedCards.appendChild(cardInfo);

        if (selectedCount === CARDS_TO_PICK) {
            resetButton.classList.remove('hidden');
            cardGrid.querySelectorAll('.card:not(.card-revealed)').forEach(card => {
                card.style.pointerEvents = 'none';
                card.classList.add('disabled');
            });
        }
    }

    // --- Event Listeners ---
    resetButton.addEventListener('click', () => {
        selectedCards.innerHTML = '';
        initializeSpread();
    });
});